import HttpClient from "../httpClient";
import DomainClient from "../domain.client";

/**
 * metaapi.cloud Token Management API client
 */
export default class TokenManagementClient {

  /**
   * Constructs token management API client instance
   * @param {HttpClient} httpClient HTTP client
   * @param {DomainClient} domainClient domain client
   */
  constructor(httpClient: HttpClient, domainClient: DomainClient);

  /**
   * Gets access rules manifest
   * @returns {Promise<Array<ManifestAccessRule>>} access rules manifest
   */
  getAccessRules(): Promise<Array<ManifestAccessRule>>

  /**
   * Returns narrowed down token with given access rules
   * @param {NarrowDownAccessRules | NarrowDownSimplifiedAccessRules} narrowDownPayload narrowed down token payload
   * @param {Number} [validityInHours] token validity in hours, default is 24 hours
   * @return {Promise<NarrowedDownToken>} promise resolving with narrowed down token
   */
  narrowDownToken(narrowDownPayload: NarrowDownAccessRules | NarrowDownSimplifiedAccessRules, validityInHours?: Number): Promise<NarrowedDownToken>
}


/**
 * Access rules manifest
 */
export declare type ManifestAccessRule = {

  /**
   * Application id
   */
  id: string;

  /**
   * Application name
   */
  application: string;

  /**
   * Application description
   */
  description: string;

  /**
   * Application resources entities
   */
  entities: Array<ManifestEntity>;

  /**
   * Application services
   */
  services: Array<ManifestService>;

  /**
   * Application method groups
   */
  methodGroups: Array<ManifestMethodGroup>;

  /**
   * Application roles
   */
  roles: Array<ManifestRoles>;

  /**
   * Application entity composition description
   */
  entityCompositionDescription?: string;
}

/**
 * Access rules manifest resource entity
 */
export declare type ManifestEntity = {

  /**
   * Entity description
   */
  description: string;

  /**
   * Entity name
   */
  entity: string;

  /**
   * Entity id description
   */
  idDescription?: string;
}

/**
 * Access rules manifest application service
 */
export declare type ManifestService = {

  /**
   * Service description
   */
  description: string;

  /**
   * Service name
   */
  service: string;
}

/**
 * Access rules manifest application method group
 */
export declare type ManifestMethodGroup = {

  /**
   * Method group name
   */
  group: string;

  /**
   * Method group description
   */
  description: string;

  /**
   * Method group methods
   */
  methods: Array<ManifestMethod>;
}

/**
 * Access rules manifest application method
 */
export declare type ManifestMethod = {

  /**
   * Method description
   */
  description: string;

  /**
   * Method name
   */
  method: string;

  /**
   * Method scopes
   */
  scopes?: Array<string>;
}

/**
 * Access rules manifest application roles
 */
export declare type ManifestRoles = {

  /**
   * Application roles description
   */
  description: string;

  /**
   * Application roles
   */
  roles: Array<string>;
}

/**
 * Narrowed down token access rules
 */
export declare type NarrowDownAccessRules = {

  /**
   * Applications access rules to grant
   */
  accessRules: Array<AccessRule>;
}

/**
 * Narrowed down token simplified access rules
 */
export declare type NarrowDownSimplifiedAccessRules = {

  /**
   * Resources to grant access to
   */
  resources?: Array<AccessRuleResource>;

  /**
   * Roles to grant access to
   */
  roles?: Array<string>

  /**
   * Applications to grant access to
   */
  applications?: Array<string>
}

/**
 * New narrowed down token model
 */
export declare type NarrowedDownToken = {

  /**
   * Authorization token value
   */
  token: string
}

/**
 * Narrowed down token access rule
 */
export declare type AccessRule = {

  /**
   * Application id to grant access to
   */
  id: string;

  /**
   * Application to grant access to
   */
  application: string;

  /**
   * Application service to grant access to
   */
  service: string;

  /**
   * Application service methodGroups to grant access to
   */
  methodGroups: Array<MethodGroups>

  /**
   * Application service resources to grant access to
   */
  resources: Array<AccessRuleResource>

  /**
   * Access rule roles to grant access to
   */
  roles: Array<string>
}

/**
 * Narrowed token access rule method groups 
 */
export declare type MethodGroups = {

  /**
   * Method group
   */
  group: string;

  /**
   * Method group methods
   */
  methods: Array<Method>;
}

/**
 * Method group method
 */
export declare type Method = {

  /**
   * Method name
   */
  method: string;

  /**
   * Method scopes
   */
  scopes?: Array<string>;
}

/**
 * Narrowed token access rule resource 
 */
export declare type AccessRuleResource = {

  /**
   * Entity name
   */
  entity: string;

  /**
   * Entity id
   */
  id: string
}