"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return TokenManagementApi;
    }
});
let TokenManagementApi = class TokenManagementApi {
    /**
   * Narrowed down token access rules
   * @typedef {Object} NarrowDownAccessRules
   * @property {Array<AccessRule>} accessRules applications access rules to grant
   */ /**
   * Narrowed down token simplified access rules
   * @typedef {Object} NarrowDownSimplifiedAccessRules
   * @property {Array<AccessRuleResource>} [resources] resources to grant access to
   * @property {Array<String>} [roles] roles to grant access to
   * @property {Array<String>} [applications] applications to grant access to
   */ /**
   * Access rules manifest
   * @typedef {Object} ManifestAccessRule
   * @property {String} id application id
   * @property {String} application application name
   * @property {String} description application description
   * @property {Array<ManifestEntity>} entities application resources entities
   * @property {Array<ManifestService>} services application services
   * @property {Array<ManifestMethodGroup>} methodGroups application method groups
   * @property {Array<ManifestRoles>} roles application roles
   * @property {String} [entityCompositionDescription] application entity composition description
   */ /**
   * Access rules manifest resource entity
   * @typedef {Object} ManifestEntity
   * @property {String} description entity description
   * @property {String} entity entity name
   * @property {String} [idDescription] entity id description
   */ /**
   * Access rules manifest application service
   * @typedef {Object} ManifestService
   * @property {String} description service description
   * @property {String} service service name
   */ /**
   * Access rules manifest application method group
   * @typedef {Object} ManifestMethodGroup
   * @property {String} group method group name
   * @property {String} description method group description
   * @property {Array<ManifestMethod>} methods method group methods
   */ /**
   * Access rules manifest application method
   * @typedef {Object} ManifestMethod
   * @property {String} description method description
   * @property {Array<String>} method method name
   * @property {Array<String>} [scopes] method scopes
   */ /**
   * Access rules manifest application roles
   * @typedef {Object} ManifestRoles
   * @property {String} description application roles description
   * @property {Array<String>} roles application roles
   */ /**
   * Narrowed down token access rule
   * @typedef {Object} AccessRule
   * @property {String} id application id to grant access to
   * @property {String} application application to grant access to
   * @property {String} service application service to grant access to
   * @property {Array<MethodGroups>} methodGroups application service methodGroups to
   * grant access to
   * @property {Array<AccessRuleResource>} resources application service resources 
   * to grant access to
   * @property {Array<String>} roles access rule roles to grant access to
   */ /**
   * Narrowed token access rule method groups 
   * @typedef {Object} MethodGroups
   * @property {String} group method group
   * @property {Array<Method>} methods method group methods
   */ /**
   * Method group method
   * @typedef {Object} Method
   * @property {String} method method
   * @property {Array<String>} [scopes] method scopes
   */ /**
   * Narrowed token access rule resource 
   * @typedef {Object} AccessRuleResource
   * @property {String} entity entity
   * @property {String} id entity id
   */ /**
   * Gets access rules manifest
   * @returns {Promise<Array<ManifestAccessRule>>} access rules manifest
   */ getAccessRules() {
        return this._tokenManagementClient.getAccessRules();
    }
    /**
   * Returns narrowed down token with given access rules
   * @param {NarrowDownAccessRules | NarrowDownSimplifiedAccessRules} narrowDownPayload narrow down payload
   * @param {Number} [validityInHours] token validity in hours, default is 24 hours
   * @returns {Promise<String>} narrowed down token
   */ async narrowDownToken(narrowDownPayload, validityInHours) {
        const narrowedToken = await this._tokenManagementClient.narrowDownToken(narrowDownPayload, validityInHours);
        return narrowedToken.token;
    }
    /**
   * Returns narrowed down token with access to given resources
   * @param {Array<AccessRuleResource>} resources resources to grant access to
   * @param {Number} [validityInHours] token validity in hours, default is 24 hours
   * @returns {Promise<String>} narrowed down token
   */ async narrowDownTokenResources(resources, validityInHours) {
        const narrowedToken = await this._tokenManagementClient.narrowDownToken({
            resources
        }, validityInHours);
        return narrowedToken.token;
    }
    /**
   * Returns narrowed down token with access to given roles
   * @param {Array<String>} roles roles to grant access to
   * @param {Number} [validityInHours] token validity in hours, default is 24 hours
   * @returns {Promise<String>} narrowed down token
   */ async narrowDownTokenRoles(roles, validityInHours) {
        const narrowedToken = await this._tokenManagementClient.narrowDownToken({
            roles
        }, validityInHours);
        return narrowedToken.token;
    }
    /**
   * Returns narrowed down token with access to given applications
   * @param {Array<String>} applications applications to grant access to
   * @param {Number} [validityInHours] token validity in hours, default is 24 hours
   * @returns {Promise<String>} narrowed down token
   */ async narrowDownTokenApplications(applications, validityInHours) {
        const narrowedToken = await this._tokenManagementClient.narrowDownToken({
            applications
        }, validityInHours);
        return narrowedToken.token;
    }
    /**
   * Checks if token resources access is restricted
   * @param {String} token token to check
   * @returns {Boolean} is token narrowed down
   */ areTokenResourcesNarrowedDown(token) {
        const parsedPayload = JSON.parse(Buffer.from(token.split(".")[1], "base64").toString());
        const areResourcesRestricted = parsedPayload.accessRules.find((rule)=>{
            return rule.resources.find((resource)=>!/^\*:\S*:\*$/.test(resource));
        });
        if (areResourcesRestricted) {
            return true;
        }
        return false;
    }
    /**
   * Constructs a TokenManagement API instance
   * @param {TokenManagementClient} tokenManagementClient tokenManagement REST API client
   */ constructor(tokenManagementClient){
        this._tokenManagementClient = tokenManagementClient;
    }
};

//# sourceMappingURL=data:application/json;base64,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